window.O || (window.O = {});
/**
 * I/O utilities.
 * @module io
 */
/**
 * Module's main object.
 * @class IO
 * @static
 */
O.IO = {
    /**
     * Read and return content of the local widget's file.
     * @method readLocalFile
     * @param {String} path
     * @return {String}
     */
    readLocalFile: function(path) {
        var content;

        new O.IO.Req({
            url: path,
            onSuccess: {
                fn: function(xhr) {
                    content = xhr.responseText;
                }
            },
            onFailure: {
                fn: function() {
                    throw new Error('Cannot read file');
                }
            },
            async: false,
            proxy: false
        });

        return content;
    }
};


/**
 * Ajax request class.
 * @class Req
 * @constructor
 */
O.IO.Req = function(o) {
    this._aborted = false;
    this._url = o.url;
    this._onSuccess = o.onSuccess;
    this._onFailure = o.onFailure;
    this._data = o.data || '';
    this._async = typeof o.async === 'boolean' ? o.async : true;
    this._headers = o.headers || [];
    this._method = o.method || O.IO.Req.GET;
    this.readFromLocalFS = !/^https?:\/\/.*/.test(this._url);
    this.useProxy = typeof o.proxy != 'undefined' ? o.proxy : true;

    if(this.useProxy) { // Use proxy
        this.useProxy = true;

        this._url = PROXY_URL + encodeURIComponent(o.url);
        if(this._method === O.IO.Req.POST) {
            this._url += '&_proxy_fields=' +
                this._getParams(this._data).join(',');
        }

        if(this._headers.length) {
            this._url += '&_proxy_headers=' + this._headers.map(function(i) {
                return i.name
            }).join(',');
        }
    }
    this._req = new XMLHttpRequest();
    this._req.open(this._method, this._url, this._async);
    this._req.setRequestHeader('User-Agent', 'O lib');

    if(this._method === O.IO.Req.POST) {
        this._req.setRequestHeader('Content-type',
                'application/x-www-form-urlencoded');
    }

    var that = this;

    this._headers.forEach(function(header) {
		that._req.setRequestHeader(header.name, header.val);
    });

    this._req.onreadystatechange = function () {
        if (this.readyState != 4 || that.isAborted()) { return };

        clearTimeout(that._timeout);

        if(that.useProxy) {
        	try {
        		var data = JSON.parse(this.responseText);
        	}
        	catch(e) {
        		that._onFailure.fn.call(that._onFailure.scope, this, {
                    code: O.IO.Req.CONNECTION_FAILURE,
                    msg: 'Connection failure'
                });
        		return;
        	}

            this.responseText2 = data.message;
            if(data.code === 200 || this.status == 304) {
                that._onSuccess.fn.call(that._onSuccess.scope, this);
            }
            else {
                that._onFailure.fn.call(that._onFailure.scope, this, {
                    code: O.IO.Req.CONNECTION_FAILURE,
                    msg: 'Connection failure'
                });
            }

            return;
        }

        if(this.status == 200 || this.status == 304 ||
                (this.status == 0 && that.readFromLocalFS)) {
            that._onSuccess.fn.call(that._onSuccess.scope, this);
        }
        else {
        	if(this.status == 0 && !that.readFromLocalFS) {
            	that._onFailure.fn.call(that._onFailure.scope, this, {
                    code: O.IO.Req.CONNECTION_FAILURE,
                    msg: 'Connection failure'
                });
            }
            else {
                that._onFailure.fn.call(that._onFailure.scope, this);
            }
        }
    };

    if(this._req.readyState == 4) { return; }
    
	try {
		this._req.send(this._data ? this._data + '' : '');
	}
	catch(e) {
        if(e.message == 'Security violation') {
            this._onFailure.fn.call(this._onFailure.scope, this, {
                code: O.IO.Req.CONNECTION_FAILURE,
                msg: 'Connection failure'
            });

            return;
        }
        else {
            throw e;
        }
	}

    if(this._async) {
        this._timeout = setTimeout(function() {
            that.abort();

            that._onFailure.fn.call(that._onFailure.scope, that, {
                code: O.IO.Req.CONNECTION_TIMEOUT,
                msg: 'Connection timeout'
            });
        }, O.IO.Req.TIMEOUT);
    }
};

/**
 * Default timeout for requests.
 * @property TIMEOUT
 * @type {Number}
 * @default 30 * 1000
 * @static
 */
O.IO.Req.TIMEOUT = 30 * 1000;

/**
 * Code for connection failure error.
 * @property CONNECTION_FAILURE
 * @type {String}
 * @default 'O.IO.Req#1'
 * @static
 */
O.IO.Req.CONNECTION_FAILURE = 'O.IO.Req#1';

/**
 * Code for connection timeout error.
 * @property CONNECTION_TIMEOUT
 * @type {String}
 * @default 'O.IO.Req#2'
 * @static
 */
O.IO.Req.CONNECTION_TIMEOUT = 'O.IO.Req#2';

/**
 * Name for GET method
 * @property GET
 * @type {String}
 * @default 'GET'
 * @static
 */
O.IO.Req.GET = 'GET';

/**
 * Name for POST method
 * @property POST
 * @type {String}
 * @default 'POST'
 * @static
 */
O.IO.Req.POST = 'POST';

/**
 * Name for DELETE method.
 * @property DELETE
 * @type {String}
 * @default 'DELETE'
 * @static
 */
O.IO.Req.DELETE = 'DELETE';


O.IO.Req.prototype = {
    // Fix constructor
    constructor: O.IO.Req,
    /**
     * Returns true if request was aborted and false otherwise.
     * @method isAborted
     * @return {Boolean}
     */
    isAborted: function() {
        return this._aborted;
    },
    /**
     * Aborts request.
     * @method abort
     */
    abort: function() {
        this._aborted = true;
        this._req.abort();
        clearTimeout(this._timeout);
    },
    /**
     * Returns response text from remote server (Without proxy part).
     * @method getRespText
     * @return {String}
     */
    getRespText: function() {
        return this._req.responseText2 || this._req.responseText;
    },
    /**
     * Return name of the parameters for request.
     * @method _getParams
     * @private
     * @return {Array}
     */
    _getParams: function(data) {
        var fields = [];

        data.split('&').forEach(function(item) {
            fields.push(item.split('=')[0]);
        });

        return fields;
    }
};
